using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// Class representing remote procedure call used in VistALink.
	/// </summary>	
	public class RpcRequest
	{
		private readonly string _rpcName;
		private readonly string _rpcSecurityContext;
		private readonly Timeout _rpcTimeout;
		private readonly RpcParameterCollection _parameters;

		// Constants used in XML serialization/deserialization
		private const string XMLCONSTS_RPCREQUEST_NODE_NAME = "Request";
		private const string XMLCONSTS_RPCREQUEST_RPCNAME_ATTRIBUTE_NAME = "rpcName";
		private const string XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_NAME = "version";
		private const string XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_VALUE = "1.0";
		private const string XMLCONSTS_RPCREQUEST_RPC_CLIENT_TIMEOUT_ATTRIBUTE_NAME = "rpcClientTimeOut";	
		private const string XMLCONSTS_RPCCONTEXT_NODE_NAME = "RpcContext";			

		/// <summary>
		/// Simplified constructor. RPC name and RPC security context must be specified.
		/// </summary>
		/// <param name="rpcName">RPC name. Arbitrary string.</param>
		/// <param name="rpcSecurityContext">RPC security context.</param>
		public RpcRequest( string rpcName, string rpcSecurityContext )
			: this( rpcName, rpcSecurityContext, null ) {}

		/// <summary>
		/// Full constructor. Specifies RPC name, RPC security context and RPC timeout. 
		/// </summary>
		/// <param name="rpcName">RPC name. Arbitrary string.</param>
		/// <param name="rpcSecurityContext">RPC security context.</param>
		/// <param name="rpcTimeout">RPC timeout. If its null, it will be initialized with default value from GlobalConfig.</param>
		public RpcRequest( string rpcName, string rpcSecurityContext, Timeout rpcTimeout )
		{
			if( rpcName == null )
				throw( new ArgumentNullException( "rpcName" ) );

			if( rpcSecurityContext == null )
				throw( new ArgumentNullException( "rpcSecurityContext" ) );
			
			_rpcName = rpcName;
			_rpcSecurityContext = rpcSecurityContext;			

			if( rpcTimeout == null )
				_rpcTimeout = GlobalConfig.DefaultRpcRequestTimeout;
			else
				_rpcTimeout = rpcTimeout;

			_parameters = new RpcParameterCollection();
		}

		/// <summary>
		/// XML deserialization constructor. It accepts element containing among its children one
		/// and only one instance of serialized RpcRequest. This constructor will parse supplied XML 
		/// and create fully initialized instance of RpcRequst class.
		/// </summary>
		/// <param name="rootElement">
		///		XML element, containing among its children one and 
		///		only one instance of serialized RpcRequest.
		///	</param>
		/// <param name="parameterFactory">Factory object parsing source XML and creating appropriate RPC parameter objects.</param>
		public RpcRequest( XmlElement rootElement, RpcParameterXmlFactory parameterFactory )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			if( parameterFactory == null )
				throw( new ArgumentNullException( "parameterFactory" ) );

			XmlElement _requestElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_RPCREQUEST_NODE_NAME );

			XmlUtility.ParseCheckRequiredAttributeValue( _requestElement, XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_NAME, XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_VALUE );

			_rpcName = XmlUtility.ParseGetRequiredAttributeValue( _requestElement, XMLCONSTS_RPCREQUEST_RPCNAME_ATTRIBUTE_NAME );
			_rpcTimeout = Timeout.ParseStringInSeconds( XmlUtility.ParseGetRequiredAttributeValue( _requestElement, XMLCONSTS_RPCREQUEST_RPC_CLIENT_TIMEOUT_ATTRIBUTE_NAME ) );

			try
			{
				 _rpcSecurityContext = VistAObfuscator.Deobfuscate( XmlUtility.ParseGetRequiredElementWithUniqueNameCDataValue( _requestElement, XMLCONSTS_RPCCONTEXT_NODE_NAME ) );
			}
			catch( ArgumentException xcp )
			{
				throw new XmlParseException( SR.Exceptions.XmlDeserializationFailedDueToInnerException( this.GetType().Name, XMLCONSTS_RPCCONTEXT_NODE_NAME ), xcp );
			}

			_parameters = RpcParameterCollection.Parse( _requestElement, parameterFactory );
		}

		/// <summary>
		/// XML serialization method. Writes out XML message representation of 
		/// RPC request to the supplied XmlWriter.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		public void WriteRpcRequestToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_RPCREQUEST_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_RPCREQUEST_RPCNAME_ATTRIBUTE_NAME, _rpcName );
			writer.WriteAttributeString( XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_NAME, XMLCONSTS_RPCREQUEST_VERSION_ATTRIBUTE_VALUE );
			writer.WriteAttributeString( XMLCONSTS_RPCREQUEST_RPC_CLIENT_TIMEOUT_ATTRIBUTE_NAME, _rpcTimeout.ToStringSeconds() );
			
			writer.WriteStartElement( XMLCONSTS_RPCCONTEXT_NODE_NAME );
			writer.WriteCData( VistAObfuscator.Obfuscate( _rpcSecurityContext ) );
			writer.WriteEndElement();
				
			_parameters.WriteCollectionToXml( writer );

			writer.WriteEndElement();
		}

		/// <summary>
		/// RPC name.
		/// </summary>
		public string RpcName
		{
			get
			{
				return _rpcName;
			}
		}

		/// <summary>
		/// RPC security context. String used as an additional
		/// guarding security measure. Each RPC resides in specific security context;
		/// caller must specify matching RPC context and name for the RPC to run.
		/// </summary>
		public string RpcSecurityContext
		{
			get
			{
				return _rpcSecurityContext;
			}
		}

		/// <summary>
		/// RPC timeout.
		/// </summary>
		public Timeout RpcTimeout
		{
			get
			{
				return _rpcTimeout;
			}
		}

		/// <summary>
		/// Collection of RPC parameters. 
		/// </summary>
		public RpcParameterCollection Parameters
		{
			get
			{
				return _parameters;
			}
		}
	}
}